<?php
session_start();
require_once '../config/db.php';
require_once '../auth/auth.php';

requireLogin();
header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(['success' => false, 'message' => 'Invalid request method']);
    exit();
}

try {
    // Validate required fields
    $required_fields = ['invoice_id', 'payment_method', 'payment_date', 'payment_amount'];
    foreach ($required_fields as $field) {
        if (!isset($_POST[$field]) || empty($_POST[$field])) {
            echo json_encode(['success' => false, 'message' => "Missing required field: $field"]);
            exit();
        }
    }

    $invoice_id = intval($_POST['invoice_id']);
    $payment_amount = floatval($_POST['payment_amount']);
    $payment_method = $_POST['payment_method'];
    $payment_date = $_POST['payment_date'];
    $reference_number = $_POST['reference_number'] ?? '';
    $payment_notes = $_POST['payment_notes'] ?? '';

    if ($payment_amount <= 0) {
        echo json_encode(['success' => false, 'message' => 'Payment amount must be greater than zero']);
        exit();
    }

    // Start transaction
    $pdo->beginTransaction();

    // Get invoice details
    $stmt = $pdo->prepare("
        SELECT i.*, c.name as customer_name 
        FROM invoices i 
        LEFT JOIN customers c ON i.customer_id = c.id 
        WHERE i.id = ?
    ");
    $stmt->execute([$invoice_id]);
    $invoice = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$invoice) {
        throw new Exception('Invoice not found');
    }

    if ($invoice['status'] === 'paid') {
        throw new Exception('Invoice is already paid');
    }

    // Allow partial payments - check if payment amount exceeds invoice total
    if ($payment_amount > $invoice['total_amount']) {
        throw new Exception('Payment amount cannot exceed invoice total. Invoice total: Ksh ' . number_format($invoice['total_amount'], 2));
    }

    // Handle M-Pesa payment if selected
    if ($payment_method === 'mpesa') {
        // Get M-Pesa settings
        $stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings WHERE setting_group = 'mpesa_settings'");
        $stmt->execute();
        $mpesaSettings = [];
        while ($row = $stmt->fetch()) {
            $mpesaSettings[$row['setting_key']] = $row['setting_value'];
        }
        
        // Check if M-Pesa is enabled and configured
        if (empty($mpesaSettings['enable_mpesa']) || $mpesaSettings['enable_mpesa'] !== '1') {
            throw new Exception('M-Pesa payment is not enabled in system settings');
        }
        
        if (empty($mpesaSettings['mpesa_consumer_key']) || empty($mpesaSettings['mpesa_consumer_secret']) || 
            empty($mpesaSettings['mpesa_shortcode']) || empty($mpesaSettings['mpesa_passkey'])) {
            throw new Exception('M-Pesa API credentials are not properly configured');
        }
        
        // For now, we'll just record the payment as M-Pesa
        // In a full implementation, you would integrate with M-Pesa API here
        $payment_notes .= " [M-Pesa Payment - API integration pending]";
    }
    

    
    // Record payment in payments table (create if doesn't exist)
    try {
        $stmt = $pdo->prepare("
            INSERT INTO payments (
                invoice_id, payment_method, payment_date, amount, 
                reference_number, notes, processed_by, created_at
            ) VALUES (?, ?, ?, ?, ?, ?, ?, NOW())
        ");
        $stmt->execute([
            $invoice_id,
            $payment_method,
            $payment_date,
            $payment_amount,
            $reference_number,
            $payment_notes,
            $_SESSION['user_id']
        ]);
    } catch (PDOException $e) {
        // If payments table doesn't exist, create it
        if (strpos($e->getMessage(), "doesn't exist") !== false) {
            $pdo->exec("
                CREATE TABLE IF NOT EXISTS payments (
                    id INT AUTO_INCREMENT PRIMARY KEY,
                    invoice_id INT NOT NULL,
                    payment_method VARCHAR(50) NOT NULL,
                    payment_date DATE NOT NULL,
                    amount DECIMAL(10,2) NOT NULL,
                    reference_number VARCHAR(100),
                    notes TEXT,
                    processed_by INT NOT NULL,
                    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                    FOREIGN KEY (invoice_id) REFERENCES invoices(id) ON DELETE CASCADE,
                    FOREIGN KEY (processed_by) REFERENCES users(id) ON DELETE RESTRICT,
                    INDEX idx_invoice_id (invoice_id),
                    INDEX idx_payment_date (payment_date)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
            ");
            
            // Try inserting again
            $stmt = $pdo->prepare("
                INSERT INTO payments (
                    invoice_id, payment_method, payment_date, amount, 
                    reference_number, notes, processed_by, created_at
                ) VALUES (?, ?, ?, ?, ?, ?, ?, NOW())
            ");
            $stmt->execute([
                $invoice_id,
                $payment_method,
                $payment_date,
                $payment_amount,
                $reference_number,
                $payment_notes,
                $_SESSION['user_id']
            ]);
        } else {
            throw $e;
        }
    }

    // Recalculate total payments for this invoice (AFTER inserting payment)
    $stmt = $pdo->prepare("
        SELECT COALESCE(SUM(amount), 0) as total_paid 
        FROM payments 
        WHERE invoice_id = ?
    ");
    $stmt->execute([$invoice_id]);
    $totalPaid = $stmt->fetchColumn();

    // Update invoice status based on new total paid
    if ($totalPaid >= $invoice['total_amount']) {
        // Full payment received - mark as paid
        $stmt = $pdo->prepare("
            UPDATE invoices 
            SET status = 'paid', 
                updated_at = NOW() 
            WHERE id = ?
        ");
        $stmt->execute([$invoice_id]);
        $statusMessage = "Invoice marked as paid";
    } else {
        // Partial payment - keep as pending
        $stmt = $pdo->prepare("
            UPDATE invoices 
            SET status = 'pending', 
                updated_at = NOW() 
            WHERE id = ?
        ");
        $stmt->execute([$invoice_id]);
        $statusMessage = "Partial payment recorded. Remaining balance: Ksh " . number_format($invoice['total_amount'] - $totalPaid, 2);
    }

    // Commit transaction
    $pdo->commit();

    echo json_encode([
        'success' => true,
        'message' => "Payment of Ksh " . number_format($payment_amount, 2) . " processed successfully for invoice " . $invoice['invoice_number'] . ". " . $statusMessage,
        'invoice_number' => $invoice['invoice_number'],
        'payment_amount' => $payment_amount,
        'total_paid' => $totalPaid,
        'amount' => $invoice['total_amount'],
        'paid' => $totalPaid,
        'remaining_balance' => $invoice['total_amount'] - $totalPaid
    ]);

} catch (Exception $e) {
    // Rollback transaction on error
    if ($pdo->inTransaction()) {
        $pdo->rollBack();
    }
    
    echo json_encode([
        'success' => false,
        'message' => 'Error processing payment: ' . $e->getMessage()
    ]);
} 